unit fROR_PatientEdit;
{
================================================================================
*
*       Package:        ROR - Clinical Case Registries
*       Date Created:   $Revision: 38 $  $Modtime: 3/06/07 2:16p $
*       Site:           Hines OIFO
*       Developers:     Andrey Andriyevskiy
*                                      
*       Description:    Generic Patient Record Editor
*
*	Notes:
*
================================================================================
*	$Archive: /CCR v1.5/Current/fROR_PatientEdit.pas $
*
* $History: fROR_PatientEdit.pas $
 * 
 * *****************  Version 38  *****************
 * User: Vhaishgavris Date: 3/08/07    Time: 1:47p
 * Updated in $/CCR v1.5/Current
 * 
 * *****************  Version 37  *****************
 * User: Vhaishgavris Date: 6/27/06    Time: 12:56p
 * Updated in $/CCR v1.5/Current
 * 
 * *****************  Version 36  *****************
 * User: Vhaishgavris Date: 5/17/06    Time: 4:52p
 * Updated in $/CCR v1.5/Current
 * 
 * *****************  Version 35  *****************
 * User: Vhaishgavris Date: 10/27/05   Time: 3:54p
 * Updated in $/CCR v1.0/Current
 * 
 * *****************  Version 34  *****************
 * User: Vhaishgavris Date: 9/07/05    Time: 3:45p
 * Updated in $/CCR v1.0/Current
 *
 * *****************  Version 33  *****************
 * User: Vhaishgavris Date: 8/20/05    Time: 2:46p
 * Updated in $/CCR v1.0/Current
 *
*
================================================================================
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, StdCtrls, ExtCtrls, ToolWin, Buttons, uROR_PatientData, ImgList,
  ActnList, Grids, uROR_Common, fROR_GenericRegistry, uROR_PatientDataLists,
  fROR_VistARegistries, uROR_Errors, uROR_CustomControls,
  uROR_CustomListView, uROR_GridView, BaseGrid, AdvGrid, AdvCGrid,
  uROR_AdvColGrid, VA508AccessibilityManager;

type
  TFormPatientEdit = class(TForm)
    Panel3: TPanel;
    Panel4: TPanel;
    pcMain: TPageControl;
    pnlHeader: TPanel;
    pnlSelectionRules: TPanel;
    pnlToolbar: TPanel;
    tsClinicalStatus: TTabSheet;
    lvSelectionRules: TCCRGridView;
    btnSave: TBitBtn;
    btnPending: TBitBtn;
    tsLocalFields: TTabSheet;
    agLocalFields: TCCRAdvColGrid;
    splLocalFields: TSplitter;
    pnlFieldDescription: TPanel;
    mDescr: TMemo;
    pnlLabResults: TPanel;
    Panel1: TPanel;
    lvSearchResults: TCCRGridView;
    Splitter1: TSplitter;
    Panel2: TPanel;
    Splitter2: TSplitter;
    cgvLabStats: TCCRGridView;
    btnDelete: TBitBtn;
    pnlPendingComment: TPanel;
    Panel6: TPanel;
    mPendingComments: TMemo;
    VA508AccessibilityManager1: TVA508AccessibilityManager;
    stxtSSNLabel: TStaticText;
    stxtSSN: TStaticText;
    stxtNameLabel: TStaticText;
    stxtName: TStaticText;
    stxtDOBLabel: TStaticText;
    stxtDOB: TStaticText;
    stxtStatusLabel: TStaticText;
    stxtStatus: TStaticText;
    stxtDODLabel: TStaticText;
    stxtDOD: TStaticText;
    stxtSelectionRules: TStaticText;
    stxtRegLabTests: TStaticText;
    stxtPendingComment: TStaticText;
    stxtFieldDescr: TStaticText;
    Splitter3: TSplitter;
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FormActivate(Sender: TObject);
    procedure agLocalFieldsCanEditCell(Sender: TObject; ARow,
      ACol: Integer; var CanEdit: Boolean);
    procedure agLocalFieldsCheckBoxClick(Sender: TObject; ACol,
      ARow: Integer; State: Boolean);
    procedure agLocalFieldsSetEditText(Sender: TObject; ACol,
      ARow: Integer; const Value: String);
    procedure agLocalFieldsSelectionChanged(Sender: TObject; ALeft, ATop,
      ARight, ABottom: Integer);
    procedure agLocalFieldsGetDisplText(Sender: TObject; ACol,
      ARow: Integer; var Value: String);
    procedure agLocalFieldsCellValidate(Sender: TObject; ACol,
      ARow: Integer; var Value: String; var Valid: Boolean);
    procedure pcMainChange(Sender: TObject);
    procedure pcMainEnter(Sender: TObject);
    procedure agLocalFieldsEnter(Sender: TObject);

  private
    fCurrFieldRow: Integer;
    fPatientData: TPatientData;
    fRegInfo:     TRegistryInfo;
    fSaveHeight: integer;

    procedure updateLocalFieldDescription;

  protected
    function  EditData(PtData: TPatientData): TReturnCode; virtual;
    procedure FocusErroneousControl(const errSeg: String; errPiece: Integer);
    function  getPatientData: TPatientData;
    function  getPendingComment: String;
    function  PopulateControls: TReturnCode; virtual;
    function  SaveData: TReturnCode;
    procedure setupReadOnly(const mode: Boolean);
    function  UpdateDataSet: TReturnCode; virtual;

    property PatientData: TPatientData          read getPatientData;
    property RegInfo: TRegistryInfo             read fRegInfo;

  public
    constructor Create(aRegInfo: TRegistryInfo; aOwner: TComponent = nil); reintroduce;
    destructor  Destroy; override;

    procedure AfterConstruction; override;
    function Edit(PtData: TPatientData; aPatientIEN: String = ''): TReturnCode;

    property SaveHeight: integer                read fSaveHeight
                                                write fSaveHeight;

  published
    procedure ApplyOptions; virtual;

  end;

implementation
{$R *.DFM}

uses
  uROR_Broker, uROR_CommonUtils, uROR_Utilities, uROR_User, fROR_Options,
  uROR_Strings, uROR_Classes, fROR_508PatientEdLocal;

const
  lfiToggle   = 0;
  lfiName     = 1;
  lfiDate     = 2;
  lfiComment  = 3;
  lfiIEN      = 4;
  lfiFieldIEN = 5;
  lfiDescr    = 6;

constructor TFormPatientEdit.Create(aRegInfo: TRegistryInfo; aOwner: TComponent);
begin
  if Assigned(aOwner) then
    inherited Create(aOwner)
  else
    inherited Create(Application);
  fRegInfo := aRegInfo;
  pcMain.ActivePageIndex := 0;
  Caption := aRegInfo.Description + ' ' + RSC00242;

  with agLocalFields do
    begin
      HideColumn(lfiDescr);
      HideColumn(lfiFieldIEN);
      HideColumn(lfiIEN);
    end;
  fCurrFieldRow := -1;
  fSaveHeight := -1;
end;

destructor TFormPatientEdit.Destroy;
begin
  fRegInfo := nil;
  inherited;
end;

procedure TFormPatientEdit.AfterConstruction;
begin
  inherited;
  fSaveHeight := pnlLabResults.Height + 23;
  ApplyOptions;
end;

procedure TFormPatientEdit.agLocalFieldsCanEditCell(Sender: TObject; ARow,
  ACol: Integer; var CanEdit: Boolean);
begin
  CanEdit := Not PatientData.ReadOnly;
end;

procedure TFormPatientEdit.agLocalFieldsCheckBoxClick(Sender: TObject;
  ACol, ARow: Integer; State: Boolean);
var
  rri: Integer;
begin
  with agLocalFields do
    if not State and (RealColIndex(ACol) = lfiToggle) then
      begin
        rri := RealRowIndex(ARow);
        AllCells[lfiDate,rri]    := '';
        AllCells[lfiComment,rri] := '';
      end;
end;

procedure TFormPatientEdit.agLocalFieldsEnter(Sender: TObject);
var i: integer;
  frm: TForm508PatientEdLocal;
  aList: TStringList;
  state: boolean;
  tmpStr: String;
begin
  if CCRScreenReaderActive then
  begin
    try
      //Build a list of strings from the data in the grid
      aList := TStringList.Create;
      for i := agLocalFields.FixedRows to agLocalFields.RowCount-1 do
      begin
        agLocalFields.GetCheckBoxState(lfiToggle, i, state);
        if state then
          tmpStr := '1' + U
        else
          tmpStr := '0' + U;
        tmpStr := tmpStr + agLocalFields.AllCells[lfiName, i] + U
                + agLocalFields.AllCells[lfiDescr, i] + U
                + agLocalFields.AllCells[lfiDate, i] + U
                + agLocalFields.AllCells[lfiComment, i] + U;
        aList.Add(tmpStr);
      end;

      frm := TForm508PatientEdLocal.Create(aList);

      //Put the data entered on the 508 form back into the grid
      if frm.Edit = mrOk then
      begin
        aList := frm.ListData;
        for i := 0 to aList.Count - 1 do
        begin
          tmpStr  := trim(piece(aList[i], U, 1));
          state := (tmpStr = '1');
          agLocalFields.SetCheckBoxState(lfiToggle, agLocalFields.FixedRows+i, state);
          tmpStr := trim(piece(aList[i], U, 4));
          if length(tmpStr) > 0 then
            tmpStr := FloatTOStr(WindowsDateToFMDate(StrToDate(tmpStr)));
          agLocalFields.AllCells[lfiDate, agLocalFields.FixedRows+i] := tmpStr;
          tmpStr  := trim(piece(aList[i], U, 5));
          agLocalFields.AllCells[lfiComment, agLocalFields.FixedRows+i] := tmpStr;
        end;
      end;
    finally
      aList.Clear;
      aList.Free;
      frm.Free;
    end;
    SendMessage(Self.Handle, WM_NEXTDLGCTL, 0, 0 );
  end;
end;

procedure TFormPatientEdit.agLocalFieldsGetDisplText(Sender: TObject; ACol,
  ARow: Integer; var Value: String);
begin
  with agLocalFields do
    if ARow >= FixedRows then
      begin
        if ACol = lfiDate then
          try
            Value := FMDateTimeStr(Value, [fmdtDateOnly,fmdtShortDate]);
          except
            Value := '';
          end;
      end;
end;

procedure TFormPatientEdit.agLocalFieldsSelectionChanged(Sender: TObject;
  ALeft, ATop, ARight, ABottom: Integer);
begin
  updateLocalFieldDescription;
end;

procedure TFormPatientEdit.agLocalFieldsSetEditText(Sender: TObject; ACol,
  ARow: Integer; const Value: String);
var
  cbs: Boolean;
begin
  with agLocalFields do
    if ARow >= FixedRows then
      begin
        if ACol <> lfiToggle then
          begin
            GetCheckBoxState(ACol, ARow, cbs);
            if Not cbs then
              begin
                SetCheckBoxState(lfiToggle, ARow, True);
                RepaintCell(DisplColIndex(lfiToggle), DisplRowIndex(ARow));
              end;
          end;
      end;
end;

procedure TFormPatientEdit.agLocalFieldsCellValidate(Sender: TObject; ACol,
  ARow: Integer; var Value: String; var Valid: Boolean);
begin
  with agLocalFields do
    if ARow >= FixedRows then
      begin
        if ACol = lfiDate then
          try
            Value := FloatTOStr(WindowsDateToFMDate(StrToDate(Value)));
          except
            Value := '';
          end;
      end;
end;

procedure TFormPatientEdit.ApplyOptions;
var
  ie: TCCRInterfaceElement;
begin
  ie := CCROptions.IE[ieReadOnly];
  cgvLabStats.Color := ie.Color;
  cgvLabStats.Font.Color := ie.FontColor;
  lvSearchResults.Color := ie.Color;
  lvSearchResults.Font.Color := ie.FontColor;
  lvSelectionRules.Color := ie.Color;
  lvSelectionRules.Font.Color := ie.FontColor;
end;

function TFormPatientEdit.Edit(PtData: TPatientData; aPatientIEN: String): TReturnCode;
begin
  Result := RC_OK;

  if aPatientIEN <> '' then
    begin
      Result := PtData.Load(aPatientIEN, True);
      if IsRCError(Result) then
        // ???
      else if PtData.ReadOnly then
        Result := PtData.CheckReadOnly(Result);
    end;

  if (Result = RC_OK) or (Result = RCW_READ_ONLY) then
    try
      Result := EditData(PtData);
    finally
      if aPatientIEN <> '' then PtData.Unlock;
    end;
end;

function TFormPatientEdit.EditData(PtData: TPatientData): TReturnCode;
begin
  Result := RC_OK;

  fPatientData := PtData;
  try

    if not PatientData.ReadOnly then
      begin
        btnDelete.Enabled := True;
        btnSave.Enabled   := True;
        btnPending.Enabled := True;
      end
    else
      begin
        btnDelete.Enabled := False;
        btnSave.Enabled   := False;
        btnPending.Enabled := False;
      end;

    PopulateControls;
    if ShowModal = mrNo then
      Result := RC_DELETE;

  finally
    fPatientData := nil;
  end;
end;

procedure TFormPatientEdit.FocusErroneousControl(const errSeg: String; errPiece: Integer);
begin
  if (errSeg = '') or (errPiece <= 0) then Exit;
  // ???
end;

procedure TFormPatientEdit.FormActivate(Sender: TObject);
begin
  if CCRScreenReaderActive then
  begin
    FocusControl(stxtSSNLabel);
  end
  else begin
    FocusControl(pcMain);
  end;
end;

procedure TFormPatientEdit.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
var
  rc: TReturnCode;
begin
  CanClose := True;
  if ModalResult <> mrCancel then
    begin
      if ModalResult = mrOk then
      begin
        PatientData.SaveType := 'C';
        PatientData.PendingComment := '';
      end
      else begin
        PatientData.SaveType := 'P';
        PatientData.PendingComment := getPendingComment();
      end;
      rc := UpdateDataSet();
      if rc = RCE_INVALID_INPUT then
        CanClose := False
      else if not IsRCError(rc) then
        if IsRCError(SaveData()) then
          CanClose := False;
    end;
end;

function TFormPatientEdit.getPatientData: TPatientData;
begin
  Result := fPatientData;
end;

function TFormPatientEdit.getPendingComment: String;
var
  i: integer;
begin
  Result := '';

  for i := 0 to mPendingComments.Lines.Count-1 do
  begin
    Result := Result + mPendingComments.Lines[i];
  end;

  Result := copy(result, 1, 100);
end;

procedure TFormPatientEdit.pcMainChange(Sender: TObject);
begin
  pcMain.HelpContext := pcMain.ActivePage.HelpContext;
end;

procedure TFormPatientEdit.pcMainEnter(Sender: TObject);
begin
  inherited;
  pcMain.HelpContext := pcMain.ActivePage.HelpContext;
end;

function TFormPatientEdit.PopulateControls: TReturnCode;
var
  i, ir, n: Integer;
  li: TCCRGridItem;
begin
  Result := RC_OK;
  with PatientData do
    begin
      //--- Patient's Data in the Form Header

      stxtDOB.Caption    := FMDateTimeStr(DOB);
      stxtName.Caption   := PatientName;
      stxtSSN.Caption    := FormatSSN(SSN);
      stxtStatus.Caption := 'N/A';
      stxtStatus.Caption := PatientStatusText(Status);

      if DOD <> '' then
        begin
          stxtDOD.Caption := FMDateTimeStr(DOD);
          stxtDODLabel.Visible := True;
          stxtDOD.Visible := True;
        end
      else
        begin
          stxtDODLabel.Visible := False;
          stxtDOD.Visible := False;
        end;
      stxtDODLabel.TabStop := stxtDOB.TabStop and stxtDODLabel.Visible;
      stxtDOD.TabStop := stxtDOB.TabStop and stxtDOD.Visible;

      //--- Clinical Status Tab

      lvSelectionRules.Items.BeginUpdate;
      try
        lvSelectionRules.Items.Clear;
        try
          n := SelectionRules.Count - 1;
          for i := 0 to n do
            with lvSelectionRules.Items.Add do
              begin
                AsString[0]  := ' ';
                AsInteger[1] := i + 1;
                AsString[2]  := SelectionRules[i].Date;
                AsString[3]  := SelectionRules[i].Description;
                AsString[4]  := SelectionRules[i].Location;
              end;
          lvSelectionRules.AlphaSort;
        except
          on E: Exception do
            MessageDlg508('', E.Message, mtError, [mbOK], 0);
        end;
      finally
        lvSelectionRules.Items.EndUpdate;
      end;

      lvSearchResults.Items.BeginUpdate;
      try
        lvSearchResults.Items.Clear;
        n := LabResults.Count - 1;
        for i := 0 to n do
        begin
          li := lvSearchResults.Items.Add;
          li.AsString[0] := LabResults[i].GroupName;
          li.AsString[1] := LabResults[i].Date;
          li.AsString[2] := LabResults[i].TestName;
          li.AsString[3] := LabResults[i].Value;
          //-- Section 508: JAWS is not working with this - so put the heading
          //   for the first cell in with the first cell's data.
          if CCRScreenReaderActive then
          begin
            li.AsString[0] := lvSearchResults.Fields[0].Caption + ' is '
              + LabResults[i].GroupName;
          end;
        end;
        lvSearchResults.AlphaSort;
      finally
        lvSearchResults.Items.EndUpdate;
      end;

      cgvLabStats.Items.BeginUpdate;
      try
        cgvLabStats.Items.Clear;
        n := LabResults.Stats.Count - 1;
        for i := 0 to n do
        begin
          li := cgvLabStats.Items.Add;
          li.AsString[0] := LabResults.Stats[i].GroupName;
          li.AsString[1] := LabResults.Stats[i].LatestValue;
          li.AsString[2] := LabResults.Stats[i].LatestDate;
          li.AsString[3] := LabResults.Stats[i].LowestValue;
          li.AsString[4] := LabResults.Stats[i].LowestDate;
          //-- Section 508: JAWS is not working with this - so put the heading
          //   for the first cell in with the first cell's data.
          if CCRScreenReaderActive then
          begin
            li.AsString[0] := lvSearchResults.Fields[0].Caption + ' is '
              + LabResults.Stats[i].GroupName;
          end;
        end;
        cgvLabStats.AlphaSort;
      finally
        cgvLabStats.Items.EndUpdate;
      end;

      mPendingComments.Lines.Clear;
      btnPending.Visible := (Status = '4');
      if Status = '4' then
      begin
        pnlLabResults.Align := alTop;
        pnlPendingComment.Visible := True;
        pnlPendingComment.Align := alClient;
        pnlLabResults.Height := SaveHeight;
        pnlLabResults.Align := alTop;
        btnSave.Caption := '&CONFIRM into registry';
        btnSave.Left := 264;
        btnSave.Width := 152;
        mPendingComments.Lines.Add(PendingComment);
        mPendingComments.Color := clWindow;
        mPendingComments.ReadOnly := ReadOnly;
      end
      else begin
        pnlPendingComment.Align := alNone;
        pnlPendingComment.Visible := False;
        pnlLabResults.Align := alClient;
        btnSave.Caption := '&Save';
        btnSave.Left := 560;
        btnSave.Width := 80;
      end;

       //--- Local Fields Tab
      agLocalFields.StartUpdate;
      try
        agLocalFields.RemoveAll;
        n := LocalFields.Count - 1;
        for i := 0 to n do
          with agLocalFields do
            begin
              AddRow;
              ir := RowCount - 1;
              GridCells[lfiIEN,ir]      := LocalFields[i].IEN;
              GridCells[lfiFieldIEN,ir] := LocalFields[i].FieldIEN;
              GridCells[lfiDescr,ir]    := LocalFields[i].Description;
              GridCells[lfiName,ir]     := LocalFields[i].Name;
              GridCells[lfiDate,ir]     := LocalFields[i].Date;
              GridCells[lfiComment,ir]  := LocalFields[i].Comment;
              SetCheckBoxState(lfiToggle, ir, LocalFields[i].Toggle);
            end;
      finally
        agLocalFields.EndUpdate;
      end;
      updateLocalFieldDescription;
      //--- If there are no local fields, hide the corresponding tab
      tsLocalFields.TabVisible := (agLocalFields.RowCount > 1);
    end;

end;

function TFormPatientEdit.SaveData: TReturnCode;
var
  errSeg: String;
  errPiece: Integer;
begin
  Result := PatientData.Save(errSeg, errPiece);
  if Result = RCE_INVALID_DATA then
    FocusErroneousControl(errSeg, errPiece);
end;

procedure TFormPatientEdit.setupReadOnly(const mode: Boolean);
var
  ie: TCCRInterfaceElement;
begin
  ie := CCROptions.IE[ieReadOnly];
  with agLocalFields do
    begin
      Color      := ie.Color;
      Font.Color := ie.FontColor;
    end;
end;

function TFormPatientEdit.UpdateDataSet: TReturnCode;
var
  i, rri: Integer;
  lf: TLocalField;
  bv: Boolean;
begin
  Result := RC_OK;

  with PatientData do
    begin
     //--- Local Fields Tab
      with agLocalFields do
        for i:=agLocalFields.RowCount-1 downto agLocalFields.FixedRows do
          begin
            rri := RealRowIndex(i);
            lf := LocalFields.Find(GridCells[lfiFieldIEN,rri]);
            if Assigned(lf) then
              begin
                GetCheckBoxState(lfiToggle, rri, bv);
                lf.Toggle  := bv;
                lf.Date    := GridCells[lfiDate,rri];
                lf.Comment := GridCells[lfiComment,rri];
              end;
          end;
    end;
end;

procedure TFormPatientEdit.updateLocalFieldDescription;
begin
  with agLocalFields do
    if RealRow <> fCurrFieldRow then
      begin
        mDescr.Clear;
        mDescr.Lines.Add(AllCells[lfiDescr,RealRow]);
        fCurrFieldRow := RealRow;
      end;
end;

end.

